function thetainit = initialvalues(params,allspots)
% This function provides initial values for the fit parameters by means of
% a centroid fit.

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands   

[Nypixels Nxpixels Ncfg] = size(allspots);
numparams = params.numparams;
model = params.model;
convpostolam = params.convpostolam;
pixelsize = params.pixelsize;
ImageSizex = Nxpixels*pixelsize/2;
ImageSizey = Nypixels*pixelsize/2;
ximag = -ImageSizex+pixelsize/2:pixelsize:ImageSizex;
yimag = -ImageSizey+pixelsize/2:pixelsize:ImageSizey;
[XImage,YImage] = meshgrid(ximag,yimag);

thetainit = zeros(Ncfg,numparams);
for jcfg = 1:Ncfg
  dummat = allspots(:,:,jcfg);
     
  % background estimate from the average value of the rim pixels.
  bgmask = dummat;
  bgmask(1,:) = 0.0;
  bgmask(:,1) = 0.0;
  bgmask(:,Nxpixels) = 0.0;
  bgmask(Nypixels,:) = 0.0;
  bgmask = dummat-bgmask;
  bg = sum(sum(bgmask))/(Nxpixels+Nypixels-2)/2;
  bg = max(bg,1e-3);
  
  dummat = dummat-bg;
  dummat = max(dummat,0);
    
  % calculation of the moments of the intensity distribution.
  Nph = sum(sum(dummat));
  M1x = sum(sum(XImage.*dummat));
  M1y = sum(sum(YImage.*dummat));
  M2x = sum(sum(XImage.^2.*dummat));
  M2y = sum(sum(YImage.^2.*dummat));
      
  % centroid estimate
  x0 = M1x/Nph;
  y0 = M1y/Nph;
  A2x = M2x-Nph*x0^2;
  A2y = M2y-Nph*y0^2;
  sigma_x = sqrt(A2x/Nph);
  sigma_y = sqrt(A2y/Nph);
  
% additional processing for finding peaks of diffraction orders and for
% finding the axial position
  switch model
    case 'standard2D'
      sigma = (sigma_x+sigma_y)/2;
    case 'astigmatic3D'
      F = (sigma_x^2-sigma_y^2)/(sigma_x^2+sigma_y^2);
      sigma = sqrt((sigma_x^2+sigma_y^2)/2);
    case 'diffractivecolor2D'
      sigma = sigma_y;
      gmask = 1-exp(-((XImage-x0).^2/sigma_x^2/2+(YImage-y0).^2/sigma_y^2/2));
      dummat = gmask.*dummat;
      dummatleft = dummat;
      dummatleft(XImage>x0)= 0;
      xleft = sum(dummatleft.*XImage)/sum(dummatleft);
      xleft = max([xleft -Nxpixels*pixelsize/2]);
      dummatright = dummat;
      dummatright(XImage<x0)= 0;
      xright = sum(dummatright.*XImage)/sum(dummatright);
      xright = min([xright Nxpixels*pixelsize/2]);
      deltax = (xright-xleft)/2;
      lambda = deltax*convpostolam;
    case 'diffractivecolor3D'
      sigma = sigma_y;
      gmask = 1-exp(-((XImage-x0).^2/sigma_x^2/2+(YImage-y0).^2/sigma_y^2/2));
      dummat = gmask.*dummat;
      dummatleft = dummat;
      dummatleft(XImage>x0)= 0;
      xleft = sum(dummatleft.*XImage)/sum(dummatleft);
      xleft = max([xleft -Nxpixels*pixelsize/2]);
      dummatright = dummat;
      dummatright(XImage<x0)= 0;
      xright = sum(dummatright.*XImage)/sum(dummatright);
      xright = min([xright Nxpixels*pixelsize/2]);
      deltax = (xright-xleft)/2;
      lambda = deltax*convpostolam;
      A2xleft = sum(sum((XImage-xleft).^2.*dummatleft));
      A2yleft = sum(sum((YImage-y0).^2.*dummatleft));
      Nphleft = sum(sum(dummatleft));
      sigsqleft = (A2xleft+A2yleft)/2/Nphleft;
%       sigsqleft = A2yleft/Nphleft;
      A2xright = sum(sum((XImage-xright).^2.*dummatright));
      A2yright = sum(sum((YImage-y0).^2.*dummatright));
      Nphright = sum(sum(dummatright));
      sigsqright = (A2xright+A2yright)/2/Nphright;
%       sigsqright = A2yright/Nphright;
      F = (sigsqleft-sigsqright)/(sigsqright+sigsqleft);
  end
  
% store estimated parameters depending on PSF model. 
  switch model
    case 'standard2D'
      thetainit(jcfg,1) = x0;
      thetainit(jcfg,2) = y0;
      thetainit(jcfg,3) = sigma;
      thetainit(jcfg,4) = Nph;
      thetainit(jcfg,5) = bg;
    case 'astigmatic3D'
      thetainit(jcfg,1) = x0;
      thetainit(jcfg,2) = y0;
      thetainit(jcfg,3) = sigma;
      thetainit(jcfg,4) = Nph;
      thetainit(jcfg,5) = bg;
      thetainit(jcfg,6) = F;
    case 'diffractivecolor2D'
      thetainit(jcfg,1) = x0;
      thetainit(jcfg,2) = y0;
      thetainit(jcfg,3) = sigma_y;
      thetainit(jcfg,4) = Nph;
      thetainit(jcfg,5) = bg;
      thetainit(jcfg,6) = lambda;
    case 'diffractivecolor3D'
      thetainit(jcfg,1) = x0;
      thetainit(jcfg,2) = y0;
      thetainit(jcfg,3) = sigma_y;
      thetainit(jcfg,4) = Nph;
      thetainit(jcfg,5) = bg;
      thetainit(jcfg,6) = lambda;
      thetainit(jcfg,7) = F;
  end
end